import { ethers, network } from 'hardhat';
import type { SignerWithAddress } from '@nomiclabs/hardhat-ethers/dist/src/signer-with-address';
import { Signers } from './types';
import { expect } from 'chai';
import * as dotenv from 'dotenv';
const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';
dotenv.config({ path: envFile, override: true });

describe('00 - Environment setup tests:', function () {
  const adminAddress = '0x979192D5280D5589FF1c3d46175d6594Ef475599';

  let signerWithAddress: SignerWithAddress[];
  const signers = {} as Signers;
  before(async function () {
    signerWithAddress = await ethers.getSigners();
    signers.admin = signerWithAddress[0];
    signers.signer1 = signerWithAddress[1];
    signers.signer2 = signerWithAddress[2];
    signers.signer3 = signerWithAddress[3];
    signers.signer4 = signerWithAddress[4];
  });

  describe('accounts', function () {
    it('network is hardhat', async function () {
      expect(await network.name).is.equal('hardhat');
    });
    it('should have at lest 4 accounts', async () => {
      expect(signerWithAddress.length).is.equal(4);
    });

    it('admin account address is known', async () => {
      expect(signers.admin.address).is.equal(adminAddress);
    });
    it('each account should have 100000000000000000000000 ETH in the accounts', async function () {
      const expactEach = async (signer: SignerWithAddress) => {
        const balance = await signer.getBalance();
        const balanceExpected = ethers.BigNumber.from(
          '10000000000000000000000',
        );
        return expect(
          ethers.BigNumber.from(balance).gt(balanceExpected),
        ).to.equal(true);
      };
      expect(signerWithAddress.length).is.equal(4);
      for (const signer of signerWithAddress) {
        await expactEach(signer);
      }
    });

    it('admin private key in tests is known', async function () {
      expect(process.env.ADMIN_PRIVATE_KEY).is.equal(
        'c48a01d8e2d79238acb5858b7451d99567a369c575b7c2a7730d51c373af35c5',
      );
    });
  });
});
